<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreatePoliciesTable extends Migration
{
    public function up(): void
    {
        Schema::create('policies', function (Blueprint $table) {
            $table->id();

            // Foreign keys to either customer or corporate_customer
            $table->unsignedBigInteger('customer_id')->nullable(); // individual customer
            $table->unsignedBigInteger('corporate_customer_id')->nullable(); // corporate customer

            // Policy Basic Info
            $table->string('policy_number')->unique();
            $table->string('policy_type'); // e.g., 'life', 'health', 'vehicle'
            $table->enum('coverage_type', ['individual', 'family', 'group'])->nullable();
            $table->string('policy_name')->nullable();
            $table->text('description')->nullable();

            // Dates
            $table->date('start_date');
            $table->date('end_date')->nullable();
            $table->date('issue_date')->nullable();
            $table->date('maturity_date')->nullable();
            $table->date('renewal_date')->nullable();

            // Premium and Payment Info
            $table->decimal('sum_assured', 20, 2)->nullable();
            $table->decimal('premium_amount', 20, 2)->nullable();
            $table->enum('premium_frequency', ['monthly', 'quarterly', 'semi-annually', 'annually'])->nullable();
            $table->enum('payment_method', ['bank_transfer', 'credit_card', 'cash', 'mobile_money'])->nullable();
            $table->boolean('auto_renew')->default(false);

            // Status
            $table->enum('status', ['active', 'inactive', 'cancelled', 'expired', 'pending'])->default('pending');
            $table->text('cancellation_reason')->nullable();

            // Beneficiaries / Attachments
            $table->integer('number_of_beneficiaries')->nullable();
            $table->json('attachments')->nullable(); // to store uploaded files like contracts

            // Additional Details
            $table->text('notes')->nullable();
            $table->enum('source', ['agent', 'branch', 'online', 'referral'])->nullable();

            // Tracking
            $table->unsignedBigInteger('created_by')->nullable();
            $table->unsignedBigInteger('updated_by')->nullable();
            $table->timestamps();
            $table->softDeletes();

            // Foreign key constraints (optional)
            $table->foreign('customer_id')->references('id')->on('customers')->onDelete('set null');
            $table->foreign('corporate_customer_id')->references('id')->on('corporate_customers')->onDelete('set null');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('policies');
    }
}
